<?php
/*
 * LaraClassifier - Classified Ads Web Application
 * Copyright (c) BeDigit. All Rights Reserved
 *
 * Website: https://laraclassifier.com
 * Author: Mayeul Akpovi (BeDigit - https://bedigit.com)
 *
 * LICENSE
 * -------
 * This software is provided under a license agreement and may only be used or copied
 * in accordance with its terms, including the inclusion of the above copyright notice.
 * As this software is sold exclusively on CodeCanyon,
 * please review the full license details here: https://codecanyon.net/licenses/standard
 */

namespace App\Http\Controllers\Api;

use App\Http\Requests\Front\AvatarRequest;
use App\Http\Requests\Front\UserPreferencesRequest;
use App\Http\Requests\Front\UserRequest;
use App\Services\Auth\App\Http\Requests\PasswordRequest;
use App\Services\Auth\App\Http\Requests\TwoFactorRequest;
use App\Services\Auth\PasswordService;
use App\Services\Auth\TwoFactorService;
use App\Services\UserService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * @group Users
 */
class UserController extends BaseController
{
	protected UserService $userService;
	protected PasswordService $passwordService;
	protected TwoFactorService $twoFactorService;
	
	/**
	 * @param \App\Services\UserService $userService
	 * @param \App\Services\Auth\PasswordService $passwordService
	 * @param \App\Services\Auth\TwoFactorService $twoFactorService
	 */
	public function __construct(
		UserService      $userService,
		PasswordService  $passwordService,
		TwoFactorService $twoFactorService
	)
	{
		parent::__construct();
		
		$this->userService = $userService;
		$this->passwordService = $passwordService;
		$this->twoFactorService = $twoFactorService;
	}
	
	/**
	 * List users
	 *
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function index(): JsonResponse
	{
		return $this->userService->getEntries();
	}
	
	/**
	 * Store user
	 *
	 * @bodyParam country_code string required The code of the user's country. Example: US
	 * @bodyParam language_code string The code of the user's spoken language. Example: en
	 * @bodyParam user_type_id int The ID of user type. Example: 1
	 * @bodyParam gender_id int The ID of gender. Example: 1
	 * @bodyParam name string required The name of the user. Example: John Doe
	 * @bodyParam photo_path file The file of user photo.
	 * @bodyParam auth_field string required The user's auth field ('email' or 'phone'). Example: email
	 * @bodyParam email string The user's email address (Required when 'auth_field' value is 'email'). Example: john.doe@domain.tld
	 * @bodyParam phone string The mobile phone number of the user (Required when 'auth_field' value is 'phone'). Example: +17656766467
	 * @bodyParam phone_country string required The user's phone number's country code (Required when the 'phone' field is filled). Example: null
	 * @bodyParam phone_hidden boolean Field to hide or show the user phone number in public. Example: 0
	 * @bodyParam username string The user's username. Example: john_doe
	 * @bodyParam password string required The user's password. Example: js!X07$z61hLA
	 * @bodyParam password_confirmation string required The confirmation of the user's password. Example: js!X07$z61hLA
	 * @bodyParam disable_comments boolean Field to disable or enable comments on the user's listings. Example: 1
	 * @bodyParam create_from_ip string required The user's IP address. Example: 127.0.0.1
	 * @bodyParam accept_terms boolean required Field to allow user to accept or not the website terms. Example: 1
	 * @bodyParam accept_marketing_offers boolean Field to allow user to accept or not marketing offers sending. Example: 0
	 * @bodyParam time_zone string The user's time zone. Example: America/New_York
	 * @bodyParam captcha_key string Key generated by the CAPTCHA endpoint calling (Required when the CAPTCHA verification is enabled from the Admin panel).
	 *
	 * @param \App\Http\Requests\Front\UserRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function store(UserRequest $request): JsonResponse
	{
		return $this->userService->store($request);
	}
	
	/**
	 * Get user
	 *
	 * @authenticated
	 *
	 * @queryParam embed string Comma-separated list of the post relationships for Eager Loading - Possible values: country,userType,gender,countPostsViews,countPosts,countSavedPosts. Example: null
	 *
	 * @urlParam id int required The user's ID. Example: 3
	 *
	 * @param $id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function show($id): JsonResponse
	{
		$params = [
			'embed' => request()->input('embed'),
		];
		
		return $this->userService->getEntry($id, $params);
	}
	
	/**
	 * Update user
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @bodyParam country_code string required The code of the user's country. Example: US
	 * @bodyParam language_code string The code of the user's spoken language. Example: en
	 * @bodyParam user_type_id int The ID of user type. Example: 1
	 * @bodyParam gender_id int The ID of gender. Example: 1
	 * @bodyParam name string required The name of the user. Example: John Doe
	 * @bodyParam photo_path file The file of user photo.
	 * @bodyParam remove_photo int Enable the user photo removal ('0' or '1'). When its value is '1' the user's photo file will be removed and the 'photo_path' column will be empty. Example: 0.
	 * @bodyParam auth_field string required The user's auth field ('email' or 'phone'). Example: email
	 * @bodyParam email string required The user's email address (Required when 'auth_field' value is 'email'). Example: john.doe@domain.tld
	 * @bodyParam phone string The mobile phone number of the user (Required when 'auth_field' value is 'phone'). Example: +17656766467
	 * @bodyParam phone_country string required The user's phone number's country code (Required when the 'phone' field is filled). Example: null
	 * @bodyParam phone_hidden boolean Field to hide or show the user phone number in public. Example: 0
	 * @bodyParam username string The user's username. Example: john_doe
	 * @bodyParam password string required The user's password. Example: js!X07$z61hLA
	 * @bodyParam password_confirmation string required The confirmation of the user's password. Example: js!X07$z61hLA
	 * @bodyParam disable_comments boolean Field to disable or enable comments on the user's listings. Example: 1
	 * @bodyParam latest_update_ip string required The user's IP address. Example: 127.0.0.1
	 * @bodyParam accept_terms boolean required Field to allow user to accept or not the website terms. Example: 1
	 * @bodyParam accept_marketing_offers boolean Field to allow user to accept or not marketing offers sending. Example: 0
	 * @bodyParam time_zone string The user's time zone. Example: America/New_York
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \App\Http\Requests\Front\UserRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function update($id, UserRequest $request): JsonResponse
	{
		return $this->userService->update($id, $request);
	}
	
	/**
	 * Delete user
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function destroy($id): JsonResponse
	{
		return $this->userService->destroy($id);
	}
	
	/**
	 * User's mini stats
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @urlParam id int required The user's ID. Example: 3
	 *
	 * @param $id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function stats($id): JsonResponse
	{
		return $this->userService->stats($id);
	}
	
	/**
	 * Update user's photo
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @bodyParam latest_update_ip string required The user's IP address. Example: 127.0.0.1
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \App\Http\Requests\Front\AvatarRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function updatePhoto($id, AvatarRequest $request): JsonResponse
	{
		return $this->userService->updatePhoto($id, $request);
	}
	
	/**
	 * Delete user's photo
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function removePhoto($id): JsonResponse
	{
		return $this->userService->removePhoto($id);
	}
	
	/**
	 * Update the user's password
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @bodyParam current_password string required The user's current password. Example: gJyuFRsGq01PJ
	 * @bodyParam password string required The user's password. Example: js!X07$z61hLA
	 * @bodyParam password_confirmation string required The confirmation of the user's password. Example: js!X07$z61hLA
	 * @bodyParam latest_update_ip string required The user's IP address. Example: 127.0.0.1
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \App\Services\Auth\App\Http\Requests\PasswordRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function changePassword($id, PasswordRequest $request): JsonResponse
	{
		return $this->passwordService->change($id, $request);
	}
	
	/**
	 * Setup user's Two-Factor Authentication
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @bodyParam two_factor_enabled boolean Enable or disable the two-factor option for the user. Example: 0
	 * @bodyParam two_factor_method string The two-factor method (Possible values: email, sms). Example: email
	 * @bodyParam phone string The user's phone number (Required when two_factor_method is set to sms). Example: null
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \App\Services\Auth\App\Http\Requests\TwoFactorRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function setupTwoFactor($id, TwoFactorRequest $request): JsonResponse
	{
		return $this->twoFactorService->setup($id, $request);
	}
	
	/**
	 * Update user's preferences
	 *
	 * @authenticated
	 * @header Authorization Bearer {YOUR_AUTH_TOKEN}
	 *
	 * @bodyParam disable_comments boolean Field to disable or enable comments on the user's listings. Example: 1
	 * @bodyParam latest_update_ip string required The user's IP address. Example: 127.0.0.1
	 * @bodyParam accept_terms boolean required Field to allow user to accept or not the website terms. Example: 1
	 * @bodyParam accept_marketing_offers boolean Field to allow user to accept or not marketing offers sending. Example: 0
	 * @bodyParam time_zone string The user's time zone. Example: America/New_York
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \App\Http\Requests\Front\UserPreferencesRequest $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function updatePreferences($id, UserPreferencesRequest $request): JsonResponse
	{
		return $this->userService->updatePreferences($id, $request);
	}
	
	/**
	 * Update the user's theme preference
	 *
	 * @bodyParam theme string required The user's theme preference value. Example: light, dark or system
	 *
	 * @urlParam id int required The user's ID. Example: 999999
	 *
	 * @param $id
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function saveThemePreference($id, Request $request): JsonResponse
	{
		return $this->userService->saveThemePreferenceForUser($id, $request);
	}
}
